<?php
/**
 * Plugin Name: ServerTrack - Conversion API & Pixel
 * Description: The complete server side tracking solution. Automatically generates DataLayer events for WooCommerce and connects to ServerTrack.io CAPI.
 * Version: 1.2.0
 * Author: Onecodesoft
 * Author URI: https://onecodesoft.com
 * License: GPL2
 * Text Domain: servertrack-pixel
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

if (!defined('SERVERTRACK_VERSION')) {
    define('SERVERTRACK_VERSION', '1.2.0');
}

/**
 * Add custom links (Docs, Support)
 */
add_filter( 'plugin_row_meta', 'servertrack_add_row_meta_links', 10, 2 );

function servertrack_add_row_meta_links( $links, $file ) {
    if ( plugin_basename( __FILE__ ) === $file ) {
        $row_meta = array(
            'docs'    => '<a href="https://servertrack.io/documentation" target="_blank" aria-label="View Documentation">Documentation</a>',
            'support' => '<a href="https://servertrack.io/support" target="_blank" aria-label="Contact Support">Support</a>',
            'guide'   => '<a href="https://servertrack.io/how-it-works" target="_blank" aria-label="How it works">How it Works</a>',
        );
        return array_merge( $links, $row_meta );
    }
    return $links;
}

class ServerTrack_Plugin {

    public function __construct() {
        // Activation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // Admin
        add_action('admin_menu', array($this, 'create_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
        
        // Frontend
        add_action('wp_enqueue_scripts', array($this, 'frontend_scripts'));
        
        // Server-side Logic
        add_action('woocommerce_add_to_cart', array($this, 'track_add_to_cart_server'), 10, 6); 
    }

    public function activate() {
        if (get_option('servertrack_mode') === false) {
            update_option('servertrack_mode', 'datalayer');
        }
        if (get_option('servertrack_subdomain') === false) {
            update_option('servertrack_subdomain', 'sdk.core-relay.org');
        }
        if (get_option('servertrack_excluded_urls') === false) {
            update_option('servertrack_excluded_urls', '');
        }
        // Default AddToCart triggers
        if (get_option('servertrack_addtocart_triggers') === false) {
            $default_triggers = array(
                array('value' => 'ajax_add_to_cart', 'type' => 'class'),
                array('value' => 'add_to_cart_button', 'type' => 'class'),
                array('value' => 'single_add_to_cart_button', 'type' => 'class'),
                array('value' => 'add-to-cart', 'type' => 'name'),
                array('value' => 'addtocart', 'type' => 'id')
            );
            update_option('servertrack_addtocart_triggers', $default_triggers);
        }
        // Default custom events (empty)
        if (get_option('servertrack_custom_events') === false) {
            update_option('servertrack_custom_events', array());
        }
    }

    public function deactivate() {
        // No cleanup required yet
    }

    // --- ADMIN ---

    public function create_menu() {
        add_options_page('ServerTrack', 'ServerTrack', 'manage_options', 'servertrack-settings', array($this, 'settings_page_html'));
    }

    public function register_settings() {
        register_setting('servertrack_group', 'servertrack_auth_key', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('servertrack_group', 'servertrack_mode', array('sanitize_callback' => array($this, 'sanitize_mode')));
        register_setting('servertrack_group', 'servertrack_subdomain', array('sanitize_callback' => array($this, 'sanitize_subdomain')));
        register_setting('servertrack_group', 'servertrack_excluded_urls', array('sanitize_callback' => array($this, 'sanitize_excluded_urls')));
        register_setting('servertrack_group', 'servertrack_addtocart_triggers', array('sanitize_callback' => array($this, 'sanitize_addtocart_triggers')));
        register_setting('servertrack_group', 'servertrack_custom_events', array('sanitize_callback' => array($this, 'sanitize_custom_events')));
    }

    public function sanitize_mode($input) {
        $valid_modes = array('datalayer', 'event', 'tracker', 'websocket');
        return in_array($input, $valid_modes, true) ? $input : 'datalayer';
    }

    public function sanitize_subdomain($input) {
        $input = sanitize_text_field($input);
        $input = preg_replace('#^https?://#', '', $input);
        $input = rtrim($input, '/');
        return empty($input) ? 'sdk.core-relay.org' : $input;
    }

    public function sanitize_excluded_urls($input) {
        // Sanitize textarea - split by lines, trim each, filter empty
        $lines = explode("\n", $input);
        $sanitized = array();
        foreach ($lines as $line) {
            $line = trim($line);
            if (!empty($line)) {
                // Allow alphanumeric, slashes, dashes, underscores, dots, query strings
                $line = preg_replace('/[^a-zA-Z0-9\/\-_.?=&:]/', '', $line);
                if (!empty($line)) {
                    $sanitized[] = $line;
                }
            }
        }
        return implode("\n", $sanitized);
    }

    /**
     * Check if current URL should be excluded from tracking
     * @return bool True if should be excluded, false otherwise
     */
    public function is_url_excluded() {
        $excluded_urls = get_option('servertrack_excluded_urls', '');
        if (empty($excluded_urls)) {
            return false;
        }

        $current_url = $_SERVER['REQUEST_URI'] ?? '';
        $current_path = parse_url($current_url, PHP_URL_PATH) ?: $current_url;
        
        // Remove leading slash for comparison
        $current_path = ltrim($current_path, '/');
        
        $excluded_lines = explode("\n", $excluded_urls);
        foreach ($excluded_lines as $excluded) {
            $excluded = trim($excluded);
            if (empty($excluded)) continue;
            
            // Remove leading slash for comparison
            $excluded = ltrim($excluded, '/');
            
            // Check if current path matches or starts with excluded path
            if ($current_path === $excluded || strpos($current_path, $excluded) === 0) {
                return true;
            }
            
            // Also check full URL if excluded entry contains domain
            if (strpos($excluded, $_SERVER['HTTP_HOST'] ?? '') !== false) {
                $full_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . ($_SERVER['HTTP_HOST'] ?? '') . $current_url;
                if (strpos($full_url, $excluded) !== false) {
                    return true;
                }
            }
        }
        
        return false;
    }

    public function sanitize_addtocart_triggers($input) {
        if (!is_array($input)) {
            return array();
        }
        $sanitized = array();
        foreach ($input as $trigger) {
            if (!isset($trigger['value']) || !isset($trigger['type'])) {
                continue;
            }
            $value = sanitize_text_field($trigger['value']);
            $type = sanitize_text_field($trigger['type']);
            
            // Validate type
            $valid_types = array('class', 'name', 'id', 'href', 'data-attribute');
            if (!in_array($type, $valid_types, true)) {
                continue;
            }
            
            // Only add if value is not empty
            if (!empty($value)) {
                $sanitized[] = array(
                    'value' => $value,
                    'type' => $type
                );
            }
        }
        return $sanitized;
    }

    public function sanitize_custom_events($input) {
        if (!is_array($input)) {
            return array();
        }
        $sanitized = array();
        foreach ($input as $event) {
            if (!isset($event['custom_url']) || !isset($event['custom_event_name'])) {
                continue;
            }
            $custom_url = sanitize_text_field($event['custom_url']);
            $custom_event_name = sanitize_text_field($event['custom_event_name']);
            $value = isset($event['value']) ? floatval($event['value']) : 0;
            
            // Only add if URL and event name are not empty
            if (!empty($custom_url) && !empty($custom_event_name)) {
                $sanitized[] = array(
                    'custom_url' => $custom_url,
                    'custom_event_name' => $custom_event_name,
                    'value' => $value
                );
            }
        }
        return $sanitized;
    }

    public function admin_scripts($hook) {
        if ($hook !== 'settings_page_servertrack-settings') {
            return;
        }
        wp_enqueue_style(
            'servertrack-admin', 
            plugins_url('assets/css/admin.css', __FILE__), 
            array(), 
            SERVERTRACK_VERSION
        );
        wp_enqueue_script(
            'servertrack-admin',
            plugins_url('assets/js/admin.js', __FILE__),
            array('jquery'),
            SERVERTRACK_VERSION,
            true
        );
    }

    public function settings_page_html() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'servertrack-pixel'));
        }
        ?>
        <div class="wrap st-wrapper">
            <div class="st-header">
                <div class="st-logo">
                    <img src="https://servertrack.io/favicon.ico" alt="ServerTrack" style="height: 32px; width: 32px; border-radius: 4px;">
                    <span>ServerTrack.io</span>
                </div>
                <div class="st-status-badge">Version <?php echo esc_html( SERVERTRACK_VERSION ); ?></div>
            </div>

            <form method="post" action="options.php">
                <?php settings_fields('servertrack_group'); ?>
                <div class="st-grid">
                    <div class="st-column-main">
                        <div class="st-card">
                            <h2><?php echo esc_html__('Configuration', 'servertrack-pixel'); ?></h2>
                            <div class="st-form-group">
                                <label class="st-label" for="servertrack_auth_key"><?php echo esc_html__('Authentication Key', 'servertrack-pixel'); ?></label>
                                <input type="text" id="servertrack_auth_key" class="st-input" name="servertrack_auth_key" value="<?php echo esc_attr(get_option('servertrack_auth_key')); ?>" placeholder="xc23ad..." />
                                <p class="st-help"><?php echo esc_html__('You can find this in your ServerTrack.io Dashboard under Server Deck > Edit Configuration > Authentication Key.', 'servertrack-pixel'); ?></p>
                            </div>
                            <div class="st-form-group">
                                <label class="st-label" for="servertrack_mode"><?php echo esc_html__('Tracking Mode', 'servertrack-pixel'); ?></label>
                                <select id="servertrack_mode" class="st-select" name="servertrack_mode">
                                    <option value="datalayer" <?php selected(get_option('servertrack_mode'), 'datalayer'); ?>><?php echo esc_html__('Full Plug-and-Play (Recommended)', 'servertrack-pixel'); ?></option>
                                    <option value="websocket" <?php selected(get_option('servertrack_mode'), 'websocket'); ?>><?php echo esc_html__('WebSocket Mode (Advanced - Bypass Blockers)', 'servertrack-pixel'); ?></option>
                                    <option value="event" <?php selected(get_option('servertrack_mode'), 'event'); ?>><?php echo esc_html__('Pixel Interceptor (Advanced)', 'servertrack-pixel'); ?></option>
                                    <option value="tracker" <?php selected(get_option('servertrack_mode'), 'tracker'); ?>><?php echo esc_html__('Developer Mode (SDK Only)', 'servertrack-pixel'); ?></option>
                                </select>
                                <p class="st-help"><?php echo esc_html__('WebSocket Mode uses dynamic paths and real-time connections to bypass ad blockers.', 'servertrack-pixel'); ?></p>
                            </div>
                            <div class="st-form-group">
                                <label class="st-label" for="servertrack_subdomain"><?php echo esc_html__('Custom Server Domain', 'servertrack-pixel'); ?></label>
                                <input type="text" id="servertrack_subdomain" class="st-input" name="servertrack_subdomain" value="<?php echo esc_attr(get_option('servertrack_subdomain', 'sdk.core-relay.org')); ?>" placeholder="track.yourdomain.com" />
                                <div class="st-alert">
                                    <strong><?php echo esc_html__('Important: CNAME-Record Cloaking Setup', 'servertrack-pixel'); ?></strong>
                                    <p style="margin:5px 0 10px;"><?php echo esc_html__('To improve tracking accuracy and bypass blockers, you should use a custom subdomain.', 'servertrack-pixel'); ?></p>
                                    <a href="https://servertrack.io/knowledge-panel/how-to-verify-your-custom-domain" target="_blank" style="text-decoration:none; color:#013D29; font-weight:600;">
                                        <?php echo esc_html__('Read the Full Setup Guide &rarr;', 'servertrack-pixel'); ?>
                                    </a>
                                </div>
                            </div>
                            <div class="st-form-group">
                                <label class="st-label" for="servertrack_excluded_urls"><?php echo esc_html__('Excluded URLs', 'servertrack-pixel'); ?></label>
                                <textarea id="servertrack_excluded_urls" class="st-textarea" name="servertrack_excluded_urls" rows="6" placeholder="/checkout&#10;/cart&#10;/my-account"><?php echo esc_textarea(get_option('servertrack_excluded_urls', '')); ?></textarea>
                                <p class="st-help"><?php echo esc_html__('Enter one URL or path per line. Tracking will be disabled on matching pages. Examples: /checkout, /cart, /privacy-policy, /terms', 'servertrack-pixel'); ?></p>
                            </div>
                        </div>

                        <!-- AddToCart Triggers Repeater -->
                        <div class="st-card" style="margin-top: 30px;">
                            <h2><?php echo esc_html__('Add To Cart - Trigger', 'servertrack-pixel'); ?></h2>
                            <p class="st-help" style="margin-bottom: 15px;"><?php echo esc_html__('Configure custom selectors to detect AddToCart buttons. Any match will trigger the event.', 'servertrack-pixel'); ?></p>
                            
                            <div id="addtocart-triggers-repeater">
                                <?php 
                                $triggers = get_option('servertrack_addtocart_triggers', array());
                                if (empty($triggers)) {
                                    $triggers = array(
                                        array('value' => 'ajax_add_to_cart', 'type' => 'class'),
                                        array('value' => 'add_to_cart_button', 'type' => 'class'),
                                        array('value' => 'single_add_to_cart_button', 'type' => 'class'),
                                        array('value' => 'add-to-cart', 'type' => 'name'),
                                        array('value' => 'addtocart', 'type' => 'id')
                                    );
                                }
                                foreach ($triggers as $index => $trigger) {
                                    ?>
                                    <div class="st-repeater-row" data-index="<?php echo esc_attr($index); ?>">
                                        <div style="display: grid; grid-template-columns: 2fr 1fr auto; gap: 10px; margin-bottom: 10px; align-items: center;">
                                            <input type="text" 
                                                   name="servertrack_addtocart_triggers[<?php echo esc_attr($index); ?>][value]" 
                                                   value="<?php echo esc_attr($trigger['value'] ?? ''); ?>" 
                                                   placeholder="ajax_add_to_cart" 
                                                   class="st-input" />
                                            <select name="servertrack_addtocart_triggers[<?php echo esc_attr($index); ?>][type]" class="st-select">
                                                <option value="class" <?php selected($trigger['type'] ?? '', 'class'); ?>>class</option>
                                                <option value="name" <?php selected($trigger['type'] ?? '', 'name'); ?>>name</option>
                                                <option value="id" <?php selected($trigger['type'] ?? '', 'id'); ?>>id</option>
                                                <option value="href" <?php selected($trigger['type'] ?? '', 'href'); ?>>href</option>
                                                <option value="data-attribute" <?php selected($trigger['type'] ?? '', 'data-attribute'); ?>>data-attribute</option>
                                            </select>
                                            <button type="button" class="st-btn-remove st-repeater-remove" style="padding: 5px 10px; background: #dc3232; color: white; border: none; cursor: pointer;">Delete</button>
                                        </div>
                                    </div>
                                    <?php
                                }
                                ?>
                            </div>
                            <button type="button" class="st-btn-add" id="add-addtocart-trigger" style="margin-top: 10px; padding: 8px 15px; background: #013D29; color: white; border: none; cursor: pointer;">Add More +</button>
                        </div>

                        <!-- Custom Events Repeater -->
                        <div class="st-card" style="margin-top: 30px;">
                            <h2><?php echo esc_html__('Custom Events', 'servertrack-pixel'); ?></h2>
                            <p class="st-help" style="margin-bottom: 15px;"><?php echo esc_html__('Track custom events on specific URL patterns.', 'servertrack-pixel'); ?></p>
                            
                            <div id="custom-events-repeater">
                                <?php 
                                $custom_events = get_option('servertrack_custom_events', array());
                                foreach ($custom_events as $index => $event) {
                                    ?>
                                    <div class="st-repeater-row" data-index="<?php echo esc_attr($index); ?>">
                                        <div style="display: grid; grid-template-columns: 2fr 2fr 1fr auto; gap: 10px; margin-bottom: 10px; align-items: center;">
                                            <input type="text" 
                                                   name="servertrack_custom_events[<?php echo esc_attr($index); ?>][custom_url]" 
                                                   value="<?php echo esc_attr($event['custom_url'] ?? ''); ?>" 
                                                   placeholder="/cart" 
                                                   class="st-input" />
                                            <input type="text" 
                                                   name="servertrack_custom_events[<?php echo esc_attr($index); ?>][custom_event_name]" 
                                                   value="<?php echo esc_attr($event['custom_event_name'] ?? ''); ?>" 
                                                   placeholder="cart" 
                                                   class="st-input" />
                                            <input type="number" 
                                                   name="servertrack_custom_events[<?php echo esc_attr($index); ?>][value]" 
                                                   value="<?php echo esc_attr($event['value'] ?? 0); ?>" 
                                                   placeholder="0" 
                                                   step="0.01"
                                                   class="st-input" />
                                            <button type="button" class="st-btn-remove st-repeater-remove" style="padding: 5px 10px; background: #dc3232; color: white; border: none; cursor: pointer;">Delete</button>
                                        </div>
                                    </div>
                                    <?php
                                }
                                ?>
                            </div>
                            <button type="button" class="st-btn-add" id="add-custom-event" style="margin-top: 10px; padding: 8px 15px; background: #013D29; color: white; border: none; cursor: pointer;">Add More +</button>
                        </div>

                        <div style="margin-top: 30px;">
                            <button type="submit" class="st-btn-primary"><?php echo esc_html__('Save Settings', 'servertrack-pixel'); ?></button>
                        </div>
                    </div>
                    <div class="st-column-sidebar">
                        <div class="st-card">
                            <h2><?php echo esc_html__('Quick Links', 'servertrack-pixel'); ?></h2>
                            <a href="https://servertrack.io/documentation" target="_blank" class="st-sidebar-link">
                                <span class="dashicons dashicons-book" style="margin-right:8px;"></span> <?php echo esc_html__('Documentation', 'servertrack-pixel'); ?>
                            </a>
                            <a href="https://servertrack.io/support" target="_blank" class="st-sidebar-link">
                                <span class="dashicons dashicons-sos" style="margin-right:8px;"></span> <?php echo esc_html__('Contact Support', 'servertrack-pixel'); ?>
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </div>
        <?php
    }

    // --- FRONTEND SCRIPTS & DATA ---

    public function frontend_scripts() {
        // Check if current URL is excluded
        if ($this->is_url_excluded()) {
            return; // Don't load any tracking scripts
        }

        $key = get_option('servertrack_auth_key');
        if (!$key) return; // Stop if no key

        // Load WebSocket SDK (/lib/{randomPath})
        $subdomain = get_option('servertrack_subdomain', 'sdk.core-relay.org');
        
        $subdomain = preg_replace('#^https?://#', '', $subdomain);
        $subdomain = rtrim($subdomain, '/');
        if (empty($subdomain)) $subdomain = 'sdk.core-relay.org';
        $api_url = "https://" . $subdomain;

        // Generate random path
        $random_path = $this->generate_random_path();
        
        // Inline script: Load /lib/{randomPath} via WebSocket
        wp_register_script('servertrack-init', '', array(), SERVERTRACK_VERSION, false);
        wp_enqueue_script('servertrack-init');
        $inline_script = sprintf(
            "(function(w, d, u, k, p) { w.ServerTrack = w.ServerTrack || {}; w.serverTrackQueue = []; w.st = function() { w.serverTrackQueue.push(arguments); }; var s = d.createElement('script'); s.async = 1; s.src = u + '/lib/' + p + '?key=' + k; var h = d.getElementsByTagName('script')[0]; h.parentNode.insertBefore(s, h); })(window, document, %s, %s, %s);",
            wp_json_encode($api_url),
            wp_json_encode($key),
            wp_json_encode($random_path)
        );
        
        wp_add_inline_script('servertrack-init', $inline_script);

        // 3. Frontend Logic (Listeners)
        if (class_exists('WooCommerce')) {
            // Load frontend.js for add-to-cart listener (works for both datalayer and websocket modes)
            wp_enqueue_script(
                'servertrack-frontend', 
                plugins_url('assets/js/frontend.js', __FILE__), 
                array('jquery', 'servertrack-init'), 
                SERVERTRACK_VERSION, 
                true
            );

            // Prepare Data for JS (currency, triggers, custom events)
            $data_for_js = array(
                'currency' => get_woocommerce_currency(),
                'cartEvents' => $this->get_cart_events_from_session(),
                'addToCartTriggers' => get_option('servertrack_addtocart_triggers', array()),
                'customEvents' => get_option('servertrack_custom_events', array())
            );

            wp_localize_script('servertrack-frontend', 'serverTrackData', $data_for_js);

            // 4. Inject Dynamic Events (Purchase, ViewContent) via Inline Script
            // Always uses WebSocket mode (window.st())
            $this->inject_dynamic_events('websocket');
        }
    }

    private function get_cart_events_from_session() {
        $cart_events = get_transient('servertrack_cart_events');
        if ($cart_events) {
            delete_transient('servertrack_cart_events');
            return $cart_events;
        }
        return array();
    }

    private function inject_dynamic_events($mode = 'datalayer') {
        // A. ViewContent
        if (is_product()) {
            global $post;
            $product = wc_get_product($post->ID);
            if ($product) {
                $event_data = array(
                    'event' => 'view_item',
                    'eventModel' => array(
                        'currency' => get_woocommerce_currency(),
                        'value' => (float) $product->get_price(),
                        'items' => array(
                            array(
                                'item_id' => (string) $product->get_id(),
                                'item_name' => $product->get_name(),
                                'price' => (float) $product->get_price(),
                                'quantity' => 1
                            )
                        )
                    )
                );
                $this->add_inline_event($event_data, $mode);
            }
        }

        // B. InitiateCheckout
        if (is_checkout() && !is_order_received_page()) {
            $cart = WC()->cart;
            if ($cart) {
                $items = array();
                foreach ($cart->get_cart() as $cart_item) {
                    $product = $cart_item['data'];
                    if (!$product) continue;
                    $items[] = array(
                        'item_id' => (string) $product->get_id(),
                        'item_name' => $product->get_name(),
                        'price' => (float) $product->get_price(),
                        'quantity' => (int) $cart_item['quantity']
                    );
                }
                $event_data = array(
                    'event' => 'begin_checkout',
                    'eventModel' => array(
                        'currency' => get_woocommerce_currency(),
                        'value' => (float) $cart->get_total('edit'),
                        'items' => $items
                    )
                );
                $this->add_inline_event($event_data, $mode);
            }
        }

        // C. Purchase
        if (is_order_received_page()) {
            global $wp;
            $order_id = isset($wp->query_vars['order-received']) ? absint($wp->query_vars['order-received']) : 0;
            if ($order_id) {
                $order = wc_get_order($order_id);
                if ($order) {
                    $items = array();
                    foreach ($order->get_items() as $item) {
                        $product = $item->get_product();
                        if (!$product) continue;
                        $item_id = $item->get_variation_id() ? (string) $item->get_variation_id() : (string) $item->get_product_id();
                        $items[] = array(
                            'item_id' => $item_id,
                            'item_name' => $item->get_name(),
                            'price' => (float) $order->get_item_total($item),
                            'quantity' => (int) $item->get_quantity()
                        );
                    }

                    $customer_data = array(
                        'first_name' => $order->get_billing_first_name(),
                        'last_name'  => $order->get_billing_last_name(),
                        'email'      => $order->get_billing_email(),
                        'phone'      => $order->get_billing_phone(),
                        'city'       => $order->get_billing_city(),
                        'zip'        => $order->get_billing_postcode(),
                        'country'    => $order->get_billing_country()
                    );

                    $event_data = array(
                        'event' => 'purchase',
                        'eventModel' => array(
                            'transaction_id' => (string) $order->get_id(),
                            'value' => (float) $order->get_total(),
                            'currency' => $order->get_currency(),
                            'tax' => (float) $order->get_total_tax(),
                            'shipping' => (float) $order->get_shipping_total(),
                            'items' => $items,
                            'customer' => array_filter($customer_data)
                        )
                    );
                    $this->add_inline_event($event_data, $mode);
                }
            }
        }
    }

    private function add_inline_event($data, $mode = 'websocket') {
        // Always use WebSocket mode: window.st('track', eventName, data, userData)
        $event_name = $this->convert_event_name_to_st($data['event']);
        $event_data = isset($data['eventModel']) ? $data['eventModel'] : array();
        $user_data = isset($data['eventModel']['customer']) ? $data['eventModel']['customer'] : array();
        
        // Get available customer data from WooCommerce session if not provided
        if (empty($user_data)) {
            $user_data = $this->get_available_customer_data();
        }
        
        // Convert WooCommerce format to ServerTrack format
        $st_data = $this->convert_woo_to_st_format($event_data);
        $st_user_data = $this->convert_customer_to_st_format($user_data);
        
        // Create inline script that calls window.st()
        $event_name_js = wp_json_encode($event_name, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        $st_data_js = wp_json_encode($st_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        $st_user_data_js = wp_json_encode($st_user_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        
        $inline_script = "(function() { if (window.st) { window.st('track', $event_name_js, $st_data_js, $st_user_data_js); } else { setTimeout(function() { if (window.st) window.st('track', $event_name_js, $st_data_js, $st_user_data_js); }, 200); } })();";
        wp_add_inline_script('servertrack-init', $inline_script);
    }

    /**
     * Convert WooCommerce event names to ServerTrack format
     */
    private function convert_event_name_to_st($woo_event) {
        $mapping = array(
            'view_item' => 'ViewContent',
            'begin_checkout' => 'InitiateCheckout',
            'purchase' => 'Purchase',
            'add_to_cart' => 'AddToCart'
        );
        return isset($mapping[$woo_event]) ? $mapping[$woo_event] : 'PageView';
    }

    /**
     * Convert WooCommerce eventModel format to ServerTrack data format
     */
    private function convert_woo_to_st_format($event_model) {
        $st_data = array();
        
        if (isset($event_model['value'])) {
            $st_data['value'] = $event_model['value'];
        }
        if (isset($event_model['currency'])) {
            $st_data['currency'] = $event_model['currency'];
        }
        if (isset($event_model['transaction_id'])) {
            $st_data['transaction_id'] = $event_model['transaction_id'];
        }
        if (isset($event_model['items'])) {
            $st_data['content_ids'] = array();
            $st_data['content_type'] = 'product';
            foreach ($event_model['items'] as $item) {
                if (isset($item['item_id'])) {
                    $st_data['content_ids'][] = $item['item_id'];
                }
            }
        }
        
        return $st_data;
    }

    /**
     * Get available customer data from WooCommerce session/customer
     * Returns data for logged-in users or customers with session data
     */
    private function get_available_customer_data() {
        $customer_data = array();
        
        if (class_exists('WooCommerce')) {
            $customer = WC()->customer;
            if ($customer) {
                // Get billing data from customer session
                $billing_country = $customer->get_billing_country();
                $billing_city = $customer->get_billing_city();
                $billing_state = $customer->get_billing_state();
                $billing_postcode = $customer->get_billing_postcode();
                $billing_email = $customer->get_email();
                $billing_phone = $customer->get_billing_phone();
                $billing_first_name = $customer->get_billing_first_name();
                $billing_last_name = $customer->get_billing_last_name();
                
                // Use billing data if available, fallback to shipping
                if (empty($billing_country)) {
                    $billing_country = $customer->get_shipping_country();
                }
                if (empty($billing_city)) {
                    $billing_city = $customer->get_shipping_city();
                }
                if (empty($billing_state)) {
                    $billing_state = $customer->get_shipping_state();
                }
                if (empty($billing_postcode)) {
                    $billing_postcode = $customer->get_shipping_postcode();
                }
                
                // Build customer data array
                if (!empty($billing_email)) {
                    $customer_data['email'] = $billing_email;
                }
                if (!empty($billing_phone)) {
                    $customer_data['phone'] = $billing_phone;
                }
                if (!empty($billing_first_name)) {
                    $customer_data['first_name'] = $billing_first_name;
                }
                if (!empty($billing_last_name)) {
                    $customer_data['last_name'] = $billing_last_name;
                }
                if (!empty($billing_city)) {
                    $customer_data['city'] = $billing_city;
                }
                if (!empty($billing_state)) {
                    $customer_data['st'] = $billing_state; // state abbreviation
                }
                if (!empty($billing_postcode)) {
                    $customer_data['zip'] = $billing_postcode;
                }
                if (!empty($billing_country)) {
                    $customer_data['country'] = $billing_country;
                }
            }
            
            // If logged in user, try to get from user meta
            if (is_user_logged_in() && empty($customer_data)) {
                $user_id = get_current_user_id();
                $customer_data['email'] = get_user_meta($user_id, 'billing_email', true) ?: get_userdata($user_id)->user_email;
                $customer_data['phone'] = get_user_meta($user_id, 'billing_phone', true);
                $customer_data['first_name'] = get_user_meta($user_id, 'billing_first_name', true) ?: get_user_meta($user_id, 'first_name', true);
                $customer_data['last_name'] = get_user_meta($user_id, 'billing_last_name', true) ?: get_user_meta($user_id, 'last_name', true);
                $customer_data['city'] = get_user_meta($user_id, 'billing_city', true);
                $customer_data['st'] = get_user_meta($user_id, 'billing_state', true);
                $customer_data['zip'] = get_user_meta($user_id, 'billing_postcode', true);
                $customer_data['country'] = get_user_meta($user_id, 'billing_country', true);
            }
        }
        
        // Filter out empty values
        return array_filter($customer_data);
    }

    /**
     * Convert customer data to ServerTrack user_data format
     */
    private function convert_customer_to_st_format($customer) {
        if (empty($customer)) return array();
        
        $st_user_data = array();
        if (isset($customer['email'])) {
            $st_user_data['em'] = $customer['email'];
        }
        if (isset($customer['phone'])) {
            $st_user_data['ph'] = $customer['phone'];
        }
        if (isset($customer['first_name'])) {
            $st_user_data['fn'] = $customer['first_name'];
        }
        if (isset($customer['last_name'])) {
            $st_user_data['ln'] = $customer['last_name'];
        }
        if (isset($customer['city'])) {
            $st_user_data['ct'] = $customer['city'];
        }
        if (isset($customer['st'])) {
            $st_user_data['st'] = $customer['st'];
        }
        if (isset($customer['zip'])) {
            $st_user_data['zp'] = $customer['zip'];
        }
        if (isset($customer['country'])) {
            $st_user_data['country'] = $customer['country'];
        }
        
        return $st_user_data;
    }

    /**
     * Generate a random path string for WebSocket mode
     * Format: 10 random alphanumeric characters
     * @return string
     */
    private function generate_random_path() {
        $characters = 'abcdefghijklmnopqrstuvwxyz0123456789';
        $path = '';
        for ($i = 0; $i < 10; $i++) {
            $path .= $characters[wp_rand(0, strlen($characters) - 1)];
        }
        return $path;
    }

    // Server-side add to cart backup (disabled for WebSocket mode - handled by frontend.js)
    public function track_add_to_cart_server($cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data) {
        // Disabled - AddToCart is now handled entirely by frontend.js via WebSocket
        return;
        
        // FIX: If this is an AJAX request, we rely on the JS listener.
        // We do NOT save a transient to avoid double-firing on next page load.
        if (wp_doing_ajax()) {
            return;
        }
        
        $product = wc_get_product($variation_id ? $variation_id : $product_id);
        if (!$product) return;
        
        $cart_events = get_transient('servertrack_cart_events') ?: array();
        $cart_events[] = array(
            'event' => 'add_to_cart',
            'eventModel' => array(
                'currency' => get_woocommerce_currency(),
                'value' => (float) $product->get_price() * $quantity,
                'items' => array(
                    array(
                        'item_id' => (string) ($variation_id ? $variation_id : $product_id),
                        'item_name' => $product->get_name(),
                        'price' => (float) $product->get_price(),
                        'quantity' => (int) $quantity
                    )
                )
            )
        );
        set_transient('servertrack_cart_events', $cart_events, 30);
    }
}

new ServerTrack_Plugin();